/*******************************************************************
*
*  DESCRIPTION: Atomic Model Decider
*
*  AUTHOR: Scott Stewart
*
*  EMAIL: wtime@live.ca
*
*  DATE: 1/3/2016
*
*******************************************************************/
/** include files **/
#include "decider.h"      		// class Line
#include "distri.h"        // class Distribution
#include "message.h"        // class ExternalMessage, InternalMessage
#include "mainsimu.h"       // MainSimulator::Instance().getParameter( ... )
#include "strutil.h"       // str2float( ... )
#include <stdlib.h>

/** public functions **/

/*******************************************************************
* Function Name: Reception
* Description:
********************************************************************/
Decider::Decider( const string &name )
: Atomic( name )
, next_person( addInputPort( "next_person" ) )
, ok( addOutputPort( "ok" ) )
, turn_away( addOutputPort( "turn_away" ) )
, extra_checks( addOutputPort( "extra_checks" ) )
, is_free( addOutputPort( "is_free" ) )
{
	id = 1;
	try
	{
		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

		MASSERT( dist );

		for ( register int i = 0 ; i < dist->varCount() ; i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
			dist->setVar( i, str2float( parameter ) ) ;
		}
	} catch( InvalidDistribution &e )
	{
		e.addText( "The model " + description() + " has distribution problems!" ) ;
		e.print(cerr);
		MTHROW( e ) ;
	} catch( MException &e )
	{
		MTHROW( e ) ;
	}
}

/*******************************************************************
* Function Name: initFunction
* Description: Resets the list
* Precondition: The time of the next internal event is infinite
********************************************************************/
Model &Decider::initFunction()
{
	passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description:
********************************************************************/
Model &Decider::externalFunction( const ExternalMessage &msg )
{
	if( msg.port() == next_person ) //New person is sent to this decider
	{
		id = static_cast < int > (msg.value());
		holdIn( active, Time( static_cast<float>( fabs( distribution().get() ) ) ) );
	}

	return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description:
********************************************************************/
Model &Decider::internalFunction( const InternalMessage & )
{
    passivate(); //When the decider has decided on the person, the decider will wait for the next person in line.
	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description:
********************************************************************/
Model &Decider::outputFunction( const InternalMessage &msg )
{
	if ((double)rand() / (double) RAND_MAX  < 0.80) // 80% chance it will be ok.
	{
		sendOutput( msg.time(), ok, id ) ;
	}else if ((double)rand() / (double) RAND_MAX  < 0.50) // 50% afterwards that it will need additional checks
	{
		sendOutput( msg.time(), extra_checks, id ) ;
	}else //Otherwise, it will be turned away
	{
		sendOutput( msg.time(), turn_away, id ) ;
	}

	sendOutput( msg.time(), is_free, id ) ; //Whatever happens, the decider will be marked as free for the line.

	return *this ;
}
