 /**
 * Canvas objects know how to render EntitySystem
 * and Environment objects. Has a private PGraphics surface
 */
class Canvas {
  PGraphics drawingSurface;
  RenderView topView;
  PGraphics ornaments; //to render non computational stuff 
                       // like debug info and such
  //RenderView customViews[];
  
  Canvas(PGraphics p, RenderView v){
    this.drawingSurface = p;
    this.topView = v;
  }

  //-------------
  // Methods 
  //--------------
  Canvas setupView(){ // pass environment bounds as paramter?
    drawingSurface.beginDraw();    
    drawingSurface.noStroke();
    drawingSurface.ortho();
    drawingSurface.background(255);
    return this; // method chaining
  }

  Canvas wrapUpView(){
    drawingSurface.endDraw();
    return this; //method chaining
  }

  Canvas beginDraw(){
    drawingSurface.beginDraw();
    return this; //method chaining
  }

  Canvas endDraw(){
    drawingSurface.endDraw();
    return this; //method chaining
  }

  Canvas drawEntitySystem(EntitySystem sys){
    Entity[] entities = sys.getEntities();
    PGraphics ds = this.drawingSurface; //shorthand
    ds.shapeMode(CORNER); ds.ellipseMode(CENTER);
    PShape entityShape = entities[0].getCollisionShape();
    PVector unit = new PVector(0,1,0);
    //entityShape = entities[0].getCollisionShape();
    for(int i = entities.length;i-->0;){ //[TIGHT_LOOP]   
      entityShape = entities[i].getCollisionShape();
      PVector pos = entities[i].getPosition(); 
      ds.pushMatrix(); 
      ds.translate(pos.x+0.5,pos.y+0.5, pos.z);
      ds.scale(CONE_RADIUS, CONE_RADIUS, 1);
      ds.shape(entityShape);
      ds.popMatrix();
    }
    return this; // method chaining
  }
  

  Canvas drawObstacles(){ 
    // [MOD] Draw custom obstacles
    modification_DrawObstacles();
    return this; // method chaining
  }

  Canvas calcEntitySystemCentroids(EntitySystem sys){
    Entity[] entities = sys.getEntities();
    PGraphics ds = this.drawingSurface; //shorthand
    int w = ds.width, h = ds.height; 
    float coneArea = PI*CONE_RADIUS*CONE_RADIUS; // sum of weights on undisturbed shape (stored in Softicle)
    int[] hits = new int[CROWDCOUNT]; // visible portion of the softicle

    // Compute centroids
    for(int i= CROWDCOUNT; i-->0;) entities[i].setCentroidPosition(new PVector());
    for (int y=0;y<ds.height;y++) { 
      for(int x=0;x<ds.width;x++) {
        int id = COLOR_TO_ID(ds.get(x,y));      // [todo] use pixels[] faster than get
        if(id < 0 || id >= CROWDCOUNT) continue;
        entities[id].getCentroidPosition().add(x,y,0);
        hits[id]++;
      }
    }
    for(int i= CROWDCOUNT; i-->0;) if(hits[i]>0) entities[i].getCentroidPosition().div(hits[i]);

    // Finally, calculate forces and integrate (solve) position
    for(int i= CROWDCOUNT; i-->0;){ //[TIGHT_LOOP] 
      Entity entity = entities[i];              // shorthand
      PVector pos = entities[i].getPosition();  // shorthand
      float x = pos.x, y=pos.y, z=pos.z;        // shorthand

      // Find centroid
      float portion = hits[i]/coneArea; entity.setPressure(1-portion);

      // Calc forces
      PVector croid = entity.getCentroidPosition(); // shorthand
      PVector voroSpring = new PVector(croid.x-x,croid.y-y, 0); 
      PVector gravity = new PVector(0,FORCE_GRAVITY,0);
      PVector friction = PVector.mult(entity.getVelocity(),-0.7);
      PVector zCompensation = new PVector(0,0, CONE_DEPTH*(0.5-portion) -z); // do relative speed
      PVector personalPath = new PVector(width/2+cos(i)*width/3-x,width/2+sin(i)*width/3-y,0).mult(0.4);
      PVector mousePath = new PVector(mouseX-x,mouseY-y,0).limit(3).mult(0.1*MOUSE_DETRACT);
      PVector bipolarPath = new PVector(0, ds.height*(i<floor(CROWDCOUNT*0.5)?0.15:0.85)-y, 0).limit(3).mult(0.1);
      PVector quadPolarPath = i>floor(CROWDCOUNT*0.75)? new PVector(width*0.82-x, width*0.82-y, 0).normalize().mult(0.95): // top left
                              i>floor(CROWDCOUNT*0.50)? new PVector(width*0.18-x, width*0.82-y, 0).normalize().mult(0.95): // top right
                              i>floor(CROWDCOUNT*0.25)? new PVector(width*0.82-x, width*0.18-y, 0).normalize().mult(0.95): // bot left
                                                        new PVector(width*0.18-x, width*0.18-y, 0).normalize().mult(0.95); // bot right
      PVector centrifugal = new PVector(ds.width/2-x, ds.height/2-y, 0);
      // explicit euler pushes tawaf outwards. Do verlet leapfrog instead.
      PVector tawaf = (new PVector(0,0,1).cross(centrifugal)).mult(0.01*centrifugal.mag()).mult(random(0.2,1)).add(centrifugal.normalize().mult(0.15));       
      
      // Net force
      PVector netForce = new PVector().add(voroSpring.mult(0.7))
                                      .add(friction)
                                      //.add(gravity.mult(0.1/**FRAME_TIME*/))
                                      //.add(zCompensation)
                                      //.add(personalPath)
                                      //.add(mousePath)
                                      //.add(bipolarPath)
                                      //.add(quadPolarPath)
                                      //.add(tawaf)
                                      ;
      if(NONSPECIAL_BIDIRECTIONAL) netForce.add(bipolarPath);
 
      
      entity.setForce(PVector.mult(netForce, 0.1)).updatePosition(); 
    }
    return this; // method chaining
  }
  
  // Compute pressure shader params
  void setupPressureShader(){
    if (!DISPLAY_PRESSURE) return; // if not displaying pressure, leave
    bwShader.set("MODE", DISPLAY_OBJECTIVE?2:1);
    bwShader.set("CROWDCOUNT", CROWDCOUNT);
    bwShader.set("COLOR_ID_STEP", COLOR_ID_STEP);
    if (!DISPLAY_OBJECTIVE) {
      float[] pressures = new float[crowd.getEntities().length];
      for (int i=crowd.getEntities().length; i-->0; )
        pressures[i] = (20+220*pow(crowd.getEntities()[i].getPressure(), 2))/255;
      bwShader.set("pressures", pressures); // change to texture in the future
    }
  }
  
  //-------------
  // Getters 
  //-------------
  PGraphics getPGraphics(){
    return this.drawingSurface;
  }
}