/*******************************************************************
*
*  DESCRIPTION: Atomic Model Scanner
*
*  AUTHOR: Guillaume Plouffe
*
*  EMAIL: mailto://gplou010@uottawa.ca
*
*  DATE: 04/03/2016
*
*******************************************************************/

/** include files **/
#include "Scanner.h"           // base header
#include "message.h"       // InternalMessage ....
#include "distri.h"        // class Distribution
#include "mainsimu.h"      // class MainSimulator


/*******************************************************************
* Function Name: Scanner
* Description: constructor
********************************************************************/
Scanner::Scanner( const string &name )
: Atomic( name )
, startIn( addInputPort( "startIn" ) )
, stopIn( addInputPort( "stopIn" ) )
, resultOut( addOutputPort( "resultOut" ) )


{

	try
	{
		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

		MASSERT( dist );

		for ( register int i = 0 ; i < dist->varCount() ; i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
			dist->setVar( i, str2float( parameter ) ) ;
		}
	}
	catch( InvalidDistribution &e )
	{
		e.addText( "The model " + description() + " has distribution problems!" ) ;
		e.print(cerr);
		MTHROW( e ) ;
	}
	catch( MException &e )
	{
		MTHROW( e ) ;
	}


}


/*******************************************************************
* Function Name: initFunction
********************************************************************/
Model &Scanner::initFunction()
{
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: This method executes when an external event is received.
********************************************************************/
Model &Scanner::externalFunction( const ExternalMessage &msg )
{
	if (this->state() == passive){
		if( msg.port() == startIn)
		{
			startTime = static_cast<float>(fabs(msg.time().asMsecs()/1000));
			holdIn(passive, Time::Zero) ;	// Wait for the product's head to be detected

		}
		else
		{
			//Ignore inputs
		}
	}
	else if (this->state() == active)
	{
		if(msg.port() == stopIn)
		{
			distributionValue = static_cast<float>( fabs( distribution().get() ) );
			stopTime = static_cast<float>(fabs(msg.time().asMsecs()/1000));
			holdIn(active, Time::Zero);
		}
	}
	
	return *this ;
}

/*******************************************************************
* Function Name: internalFunction
* Description: This method executes when the TA has expired, right after the outputFunction has finished.
* 			   The new state and TA should be set.
********************************************************************/
Model &Scanner::internalFunction( const InternalMessage & )
{

	if(this->state() == passive){

		holdIn(active, Time::Inf);

	}
	else{

		  this-> passivate();	// Set to passive when the product's head reach the scanner
	}

	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: This method executes when the TA has expired. After this method the internalFunction is called.
*              Output values can be send through output ports
********************************************************************/
Model &Scanner::outputFunction( const InternalMessage &msg )
{

	if (this->state() == passive)
	{

		// Nothing to send
	}
	else if (this->state() == active)
	{
		float meanGoodProb = exp((stopTime-startTime)*-0.0693);
		cout << msg.time()<<" Mean value= "<< meanGoodProb<<", "<< " Distribution value = "<< distributionValue -1<<endl;
		scanResult = meanGoodProb+(distributionValue-1) > 0.5 ? 1: 0;	// For a product length of 200 cm at 20 cm/s, the mean distribution is calculated to be about 0.5
		sendOutput(	msg.time(), resultOut, scanResult ) ;		// Command the scanner to start
	}
	return *this;

}

Scanner::~Scanner()
{
	//TODO: add destruction code here. Free distribution memory, etc. 
}
