/*******************************************************************
*
*  DESCRIPTION: Atomic Model HeadDetection
*
*  AUTHOR: Guillaume Plouffe
*
*  EMAIL: mailto://gplou010@uottawa.ca
*
*  DATE: 04/03/2016
*
*******************************************************************/

/** include files **/
#include "HeadDetection.h"           // base header
#include "message.h"       // InternalMessage ....
#include "distri.h"        // class Distribution
#include "mainsimu.h"      // class MainSimulator


/*******************************************************************
* Function Name: HeadDetection
* Description: constructor
********************************************************************/
HeadDetection::HeadDetection( const string &name )
: Atomic( name )
, speedIn( addInputPort( "speedIn" ) )
, productIn( addInputPort( "productIn" ) )
, scanStart( addOutputPort( "scanStart" ) )
, out( addOutputPort( "out" ) )

{

	speed = 20;	// In cm/s

	string distFromScan( MainSimulator::Instance().getParameter( description(), "distFromScanner" ) ) ;

	if( distFromScan != "" )
	{
		distFromScanner = str2Int(distFromScan) ;
		cout << "distFromScanner = " << distFromScanner << endl;
	}

	try
	{
		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

		MASSERT( dist );

		for ( register int i = 0 ; i < dist->varCount() ; i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
			dist->setVar( i, str2float( parameter ) ) ;
		}
	}
	catch( InvalidDistribution &e )
	{
		e.addText( "The model " + description() + " has distribution problems!" ) ;
		e.print(cerr);
		MTHROW( e ) ;
	}
	catch( MException &e )
	{
		MTHROW( e ) ;
	}


}

/*******************************************************************
* Function Name: initFunction
********************************************************************/
Model &HeadDetection::initFunction()
{
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: This method executes when an external event is received.
********************************************************************/
Model &HeadDetection::externalFunction( const ExternalMessage &msg )
{

	if (this->state() == passive){
		if( msg.port() == productIn)
		{

			float distributionValue = static_cast<float>( fabs( distribution().get() ) );

			holdIn(passive, Time( distributionValue )) ;	// Wait for the product's head to be detected in seconds

		}
		else if (msg.port() == speedIn)
		{
			speed = static_cast < int > (msg.value());
		}
		else
		{
			//Ignore inputs
		}
	}
	else if (this->state() == active)
	{
		// Ignore input
	}
	
	return *this ;
}

/*******************************************************************
* Function Name: internalFunction
* Description: This method executes when the TA has expired, right after the outputFunction has finished.
* 			   The new state and TA should be set.
********************************************************************/
Model &HeadDetection::internalFunction( const InternalMessage & )
{
	//TODO: implement the internal function here
	if(this->state() == passive){

		//cout << "timeToReachScan =  "<< static_cast<float>( fabs(distFromScanner/speed))<< endl;

		holdIn(active, Time(static_cast<float>( fabs(distFromScanner/speed)))) ;	// Set to active when the product's head is detected

	}
	else{

		  this-> passivate();	// Set to passive when the product's head reach the scanner

	}

	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: This method executes when the TA has expired. After this method the internalFunction is called.
*              Output values can be send through output ports
********************************************************************/
Model &HeadDetection::outputFunction( const InternalMessage &msg )
{
	//TODO: implement the output function here
	// remember you can use sendOutput(time, outputPort, value) function.
	// sendOutput( msg.time(), out, 1) ;
	
	if (this->state() == passive)
	{

		sendOutput(	msg.time(), out,  1) ;		// Warn the tail detection subnet that a product is detected
	}
	else if (this->state() == active)
	{

		sendOutput(	msg.time(), scanStart, 1 ) ;	// Command the scanner to start
	}
	return *this;

}

HeadDetection::~HeadDetection()
{
	//TODO: add destruction code here. Free distribution memory, etc. 
}
